export { renderers } from '../../renderers.mjs';

function extractVideoId(url) {
  const patterns = [
    /tiktok\.com\/@[\w.-]+\/video\/(\d+)/,
    /tiktok\.com\/v\/(\d+)/,
    /vm\.tiktok\.com\/(\w+)/,
    /tiktok\.com\/t\/(\w+)/
  ];
  for (const pattern of patterns) {
    const match = url.match(pattern);
    if (match) {
      return match[1];
    }
  }
  return null;
}
async function resolveShortUrl(url) {
  try {
    const response = await fetch(url, {
      method: "HEAD",
      redirect: "follow",
      headers: {
        "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36"
      }
    });
    return response.url;
  } catch {
    return url;
  }
}
async function fetchVideoInfoWeb(url) {
  if (url.includes("vm.tiktok.com") || url.includes("tiktok.com/t/")) {
    url = await resolveShortUrl(url);
  }
  const videoId = extractVideoId(url);
  if (!videoId) {
    throw new Error("Could not extract video ID from URL");
  }
  const apiUrl = `https://www.tikwm.com/api/?url=${encodeURIComponent(url)}`;
  const response = await fetch(apiUrl, {
    method: "GET",
    headers: {
      "Accept": "application/json",
      "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36"
    }
  });
  if (!response.ok) {
    throw new Error("Failed to fetch video information");
  }
  const result = await response.json();
  if (result.code !== 0 || !result.data) {
    throw new Error(result.msg || "Failed to process video");
  }
  const data = result.data;
  if (data.is_private) {
    throw new Error("This video is private and cannot be downloaded");
  }
  return {
    id: data.id || videoId,
    title: data.title || "",
    description: data.title || "",
    duration: data.duration || 0,
    thumbnail: data.cover || data.origin_cover || "",
    cover: data.origin_cover || data.cover || "",
    author: {
      unique_id: data.author?.unique_id || "",
      username: data.author?.unique_id || "",
      nickname: data.author?.nickname || "",
      avatar: data.author?.avatar || ""
    },
    statistics: {
      play_count: data.play_count || 0,
      like_count: data.digg_count || 0,
      comment_count: data.comment_count || 0,
      share_count: data.share_count || 0
    },
    video: {
      no_watermark: data.play || data.hdplay || "",
      watermark: data.wmplay || "",
      size: data.size || void 0,
      resolution: data.hd_size ? "HD" : "SD"
    },
    music: {
      title: data.music_info?.title || data.music || "Original Sound",
      author: data.music_info?.author || "",
      url: data.music || data.music_info?.play || ""
    }
  };
}
async function fetchVideoInfoAlternative(url) {
  if (url.includes("vm.tiktok.com") || url.includes("tiktok.com/t/")) {
    url = await resolveShortUrl(url);
  }
  const videoId = extractVideoId(url);
  if (!videoId) {
    throw new Error("Could not extract video ID from URL");
  }
  const apiUrl = "https://ssstik.io/abc?url=dl";
  const formData = new URLSearchParams();
  formData.append("id", url);
  formData.append("locale", "en");
  formData.append("tt", "aGpxWE5x");
  try {
    const response = await fetch(apiUrl, {
      method: "POST",
      headers: {
        "Content-Type": "application/x-www-form-urlencoded",
        "User-Agent": "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36",
        "Origin": "https://ssstik.io",
        "Referer": "https://ssstik.io/en"
      },
      body: formData.toString()
    });
    if (!response.ok) {
      throw new Error("Alternative API failed");
    }
    const html = await response.text();
    const noWatermarkMatch = html.match(/href="([^"]+)"[^>]*>Without watermark/);
    const watermarkMatch = html.match(/href="([^"]+)"[^>]*>With watermark/);
    const audioMatch = html.match(/href="([^"]+)"[^>]*>Download MP3/);
    if (!noWatermarkMatch && !watermarkMatch) {
      throw new Error("Could not extract download links");
    }
    return {
      id: videoId,
      title: "TikTok Video",
      description: "TikTok Video",
      duration: 0,
      thumbnail: "",
      cover: "",
      author: {
        unique_id: "unknown",
        username: "unknown",
        nickname: "TikTok User",
        avatar: ""
      },
      statistics: {
        play_count: 0,
        like_count: 0,
        comment_count: 0,
        share_count: 0
      },
      video: {
        no_watermark: noWatermarkMatch?.[1] || "",
        watermark: watermarkMatch?.[1] || ""
      },
      music: {
        title: "Original Sound",
        author: "",
        url: audioMatch?.[1] || ""
      }
    };
  } catch {
    throw new Error("All API methods failed");
  }
}
const POST = async ({ request }) => {
  try {
    const body = await request.json();
    const { url } = body;
    if (!url || typeof url !== "string") {
      return new Response(
        JSON.stringify({ error: "Please provide a valid TikTok URL" }),
        { status: 400, headers: { "Content-Type": "application/json" } }
      );
    }
    const validPatterns = [
      /^https?:\/\/(www\.)?tiktok\.com\/@[\w.-]+\/video\/\d+/,
      /^https?:\/\/(www\.)?tiktok\.com\/t\/\w+/,
      /^https?:\/\/vm\.tiktok\.com\/\w+/,
      /^https?:\/\/(www\.)?tiktok\.com\/v\/\d+/
    ];
    const isValid = validPatterns.some((pattern) => pattern.test(url));
    if (!isValid) {
      return new Response(
        JSON.stringify({ error: "Invalid TikTok URL format" }),
        { status: 400, headers: { "Content-Type": "application/json" } }
      );
    }
    let videoInfo;
    try {
      videoInfo = await fetchVideoInfoWeb(url);
    } catch (error) {
      try {
        videoInfo = await fetchVideoInfoAlternative(url);
      } catch {
        throw error;
      }
    }
    if (!videoInfo.video.no_watermark && !videoInfo.video.watermark) {
      return new Response(
        JSON.stringify({ error: "Could not retrieve download links. The video may be private or unavailable." }),
        { status: 404, headers: { "Content-Type": "application/json" } }
      );
    }
    return new Response(
      JSON.stringify(videoInfo),
      {
        status: 200,
        headers: {
          "Content-Type": "application/json",
          "Cache-Control": "no-cache"
        }
      }
    );
  } catch (error) {
    console.error("Download API Error:", error);
    if (error.message?.includes("private")) {
      return new Response(
        JSON.stringify({ error: "This video is private and cannot be downloaded" }),
        { status: 403, headers: { "Content-Type": "application/json" } }
      );
    }
    if (error.message?.includes("not found") || error.message?.includes("unavailable")) {
      return new Response(
        JSON.stringify({ error: "Video not found or no longer available" }),
        { status: 404, headers: { "Content-Type": "application/json" } }
      );
    }
    return new Response(
      JSON.stringify({ error: error.message || "Failed to process video. Please try again." }),
      { status: 500, headers: { "Content-Type": "application/json" } }
    );
  }
};
const GET = async () => {
  return new Response(
    JSON.stringify({
      status: "ok",
      message: "TikTok Downloader API",
      usage: 'POST /api/download with { url: "tiktok_video_url" }'
    }),
    { status: 200, headers: { "Content-Type": "application/json" } }
  );
};

const _page = /*#__PURE__*/Object.freeze(/*#__PURE__*/Object.defineProperty({
  __proto__: null,
  GET,
  POST
}, Symbol.toStringTag, { value: 'Module' }));

const page = () => _page;

export { page };
